using System;
using System.Data;
using System.Collections;
using System.Reflection;
using TABLE = gov.va.med.vbecs.Common.VbecsTables;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;


namespace gov.va.med.vbecs.BOL
{
		#region Header

		///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
		///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
		///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/25/2003</CreationDate>
		///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
		///<summary>
		///Static class with common antigen typing functions
		///</summary>

		#endregion

	public class AntigenTyping
	{
		#region Methods

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/25/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3268"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Hash table with interpretation for antigen typing test entries</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="225"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///	
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Return hashtable with the TABLE.BloodUnitAntigen.Positive, "Negative" and "Inconclusive" values of each antigen typing test result
		/// Implements BR_41.09
		/// </summary>
		/// <returns>Hash table</returns>
		public static Hashtable InitializeLookup()
		{
			//
			//
			//BR_41.09
			//
			Hashtable resultLookup = new Hashtable();

			// Empty results
			resultLookup.Add(string.Empty, "X");
			resultLookup.Add("X", "X");
			
			// Positive results	
			resultLookup.Add("1", "P");
			resultLookup.Add("2", "P");
			resultLookup.Add("3", "P");
			resultLookup.Add("4", "P");
			resultLookup.Add("P", "P");

			// Negative results
			resultLookup.Add("R", "N");
			resultLookup.Add("0", "N");
			resultLookup.Add("N", "N");

			//Inconclusive results
			//CR 2653 Replaced Inconclusive results with positive results
			resultLookup.Add("W", "P");
			resultLookup.Add("F", "P");
			resultLookup.Add("M", "P");
			//
			return resultLookup;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/25/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3270"> 
		///		<ExpectedInput>Data rows containing valid antigen typing test results</ExpectedInput>
		///		<ExpectedOutput>System interpretation of test results</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3271"> 
		///		<ExpectedInput>Data row with invalid column names</ExpectedInput>
		///		<ExpectedOutput>FormatException thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Return hashtable with the TABLE.BloodUnitAntigen.Positive and Negative values of each antigen typing test result
		/// Implements TT_41.01, BR_41.19, BR_2.39
		/// </summary>
		/// <param name="dr">Antigen Typing test row</param>
		/// <returns>Test interpretations: "P", "N", "X" or ""</returns>
		public static string InterpretPhenotypeResults(DataRow dr)
		{
			if ((!dr.Table.Columns.Contains(ARTIFICIAL.IS))||
				(!dr.Table.Columns.Contains(ARTIFICIAL.RT))||
				(!dr.Table.Columns.Contains(ARTIFICIAL.C37))||
				(!dr.Table.Columns.Contains(ARTIFICIAL.AHG))||
				(!dr.Table.Columns.Contains(ARTIFICIAL.CC))||
				(!dr.Table.Columns.Contains(TABLE.ReagentTestParameter.ReactivityPhaseCode)))
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InsufficientColumns(
						MethodBase.GetCurrentMethod().Name).ResString);
			}
			//
			if (((Common.Utility.GetReactivityPhaseTextFromCode(dr[TABLE.RackLot.ReactivityPhaseCode].ToString()).IndexOf(ARTIFICIAL.IS)>-1) && (dr[ARTIFICIAL.IS].ToString().Length==0))||
				((Common.Utility.GetReactivityPhaseTextFromCode(dr[TABLE.RackLot.ReactivityPhaseCode].ToString()).IndexOf(ARTIFICIAL.RT)>-1) && (dr[ARTIFICIAL.RT].ToString().Length==0))||
				((Common.Utility.GetReactivityPhaseTextFromCode(dr[TABLE.RackLot.ReactivityPhaseCode].ToString()).IndexOf(ARTIFICIAL.C37)>-1) && (dr[ARTIFICIAL.C37].ToString().Length==0))||
				((Common.Utility.GetReactivityPhaseTextFromCode(dr[TABLE.RackLot.ReactivityPhaseCode].ToString()).IndexOf(ARTIFICIAL.AHG)>-1) && (dr[ARTIFICIAL.AHG].ToString().Length==0))||
				((Common.Utility.GetReactivityPhaseTextFromCode(dr[TABLE.RackLot.ReactivityPhaseCode].ToString()).IndexOf(ARTIFICIAL.AHG)>-1) && (dr[ARTIFICIAL.CC].ToString().Length==0)))
			{
				//Can't enter an interp, if the phases aren't complete
				return string.Empty;
			}
			Hashtable resultLookup = InitializeLookup();
			//
			string IS = resultLookup[dr[ARTIFICIAL.IS].ToString()].ToString();
			string RT = resultLookup[dr[ARTIFICIAL.RT].ToString()].ToString();
			string ThreeSeven = resultLookup[dr[ARTIFICIAL.C37].ToString()].ToString();
			string AHG = resultLookup[dr[ARTIFICIAL.AHG].ToString()].ToString();
			string CC = resultLookup[dr[ARTIFICIAL.CC].ToString()].ToString();
			Common.ReactivityPhase reactivity;
			
			if (dr.IsNull(TABLE.ReagentTestParameter.ReactivityPhaseCode))
			{
				reactivity = Common.ReactivityPhase.Unknown;
			}
			else
			{
				reactivity = (Common.ReactivityPhase)Convert.ToInt32(dr[TABLE.ReagentTestParameter.ReactivityPhaseCode]);
			}
			//
			//TT_41.01
			//CR 2653 Removed all occurences of inconclusive test results and interpretations
			if ( reactivity == Common.ReactivityPhase.IS)
			{
				//IS only
				if (IS == "N")
				{
					return "N";
				}
				else if (IS == "P")
				{
					return "P";
				}
				else
				{
					return "X";
				}
			}
			else if (reactivity == Common.ReactivityPhase.ISandRT)
			{
				//IS/RT
				if ((IS == "N" || IS == "X") && RT == "N")
				{
					return "N";
				}
				else if ((IS == "P" || IS == "N" || IS == "X") && RT == "P")
				{
					return "P";
				}
				//CR 2679
				else if (IS == "P" && RT == "X")
				{
					return "P";
				}
				else
				{
					return "X";
				}
			}
			else if (reactivity == Common.ReactivityPhase.ISand37)
			{
				//IS/37
				if ((IS == "N" || IS == "X") && ThreeSeven == "N")
				{
					return "N";
				}
				//CR 2679 - added IS == "X" as a valid result for interpretation of P
				else if ((IS == "P" || IS == "N" || IS == "X") && ThreeSeven == "P")
				{
					return "P";
				}
				else if (IS == "P" && ThreeSeven == "X")
				{
					return "P";
				}
				else
				{
					return "X";
				}
			}
			else if (reactivity == Common.ReactivityPhase.AHGandCC)
			{
				//AHG/CC
				if (AHG == "N" && CC == "P")
				{
					return "N";
				}
				else if (AHG == "P" && (CC == "P" || CC == "X"))
				{
					return "P";
				}
				else
				{
					return "X";
				}
			}
			//
			//incomplete testing
			return string.Empty;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/19/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3272"> 
		///		<ExpectedInput>Valid Weak D typing test results in all valid combinations</ExpectedInput>
		///		<ExpectedOutput>System interpretation of test results</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3273"> 
		///		<ExpectedInput>Invalid Weak D typing test results ('Q')</ExpectedInput>
		///		<ExpectedOutput>Null Reference Exception thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Find the interpreatation of testing results for WeakD
		/// Implements BR_41.24, TT_41.02, BR_2.39
		/// </summary>
		/// <param name="antiD37">Anti-D 37 degree test result</param>
		/// <param name="dCtrl37">D Control 37 degree test result</param>
		/// <param name="antiDAHG">Anti-D AHG test result</param>
		/// <param name="dCtrlAHG">D Control AHG test result</param>
		/// <param name="antiDCC">Anti-D CC test result</param>
		/// <param name="dCtrlCC">D Control CC test result</param>
		/// <param name="interp">Anti-D test interpretation</param>
		/// <param name="includeDCtrl">D Control test interpretation</param>
		/// <param name="antiDInterp">Anti-D Interpretation</param>
		/// <param name="dCtrlInterp">D Control Interpretation</param>
		/// <returns></returns>
		public static string InterpretWeakDResults(string antiD37, string dCtrl37, string antiDAHG, string dCtrlAHG, string antiDCC, string dCtrlCC, string interp, bool includeDCtrl, ref string antiDInterp, ref string dCtrlInterp)
		{
			if (interp.Trim().Length==0)
			{
				return string.Empty;
			}
			//
			//
			Hashtable resultLookup = InitializeLookup();
			//
			antiD37 = resultLookup[antiD37].ToString().Trim();
			antiDAHG = resultLookup[antiDAHG].ToString().Trim();
			antiDCC = resultLookup[antiDCC].ToString().Trim();
			dCtrl37 = resultLookup[dCtrl37].ToString().Trim();
			dCtrlAHG = resultLookup[dCtrlAHG].ToString().Trim();
			dCtrlCC = resultLookup[dCtrlCC].ToString().Trim();
			//
			if (antiD37 == "P" && antiDAHG == "X" && antiDCC == "X")
			{
				antiDInterp = "P";
			}
			else if (antiD37 == "X" && antiDAHG == "P" && (antiDCC == "X" || antiDCC == "P"))
			{
				antiDInterp = "P";
			}
			else if (antiD37 == "P" && antiDAHG == "P" && (antiDCC == "X" || antiDCC == "P"))
			{
				antiDInterp = "P";
			}
			else if (antiD37 == "N" && antiDAHG == "P" && (antiDCC == "P" || antiDCC == "X"))
			{
				antiDInterp = "P";
			}
			else if ((antiD37 == "N" || antiD37 == "X") && antiDAHG == "N" && antiDCC == "P")
			{
				antiDInterp = "N";
			}
			else
			{
				antiDInterp = "X";
			}
			//
			if (dCtrl37 == "P" && dCtrlAHG == "X" && dCtrlCC == "X")
			{
				dCtrlInterp = "P";
			}
			else if (dCtrl37 == "X" && dCtrlAHG == "P" && (dCtrlCC == "X" || dCtrlCC == "P"))
			{
				dCtrlInterp = "P";
			}
			else if (dCtrl37 == "P" && dCtrlAHG == "P" && (dCtrlCC == "X" || dCtrlCC == "P"))
			{
				dCtrlInterp = "P";
			}
			else if (dCtrl37 == "N" && dCtrlAHG == "P" && (dCtrlCC == "P" || dCtrlCC == "X"))
			{
				dCtrlInterp = "P";
			}
			else if ((dCtrl37 == "N" || dCtrl37 == "X") && dCtrlAHG == "N" && dCtrlCC == "P")
			{
				dCtrlInterp = "N";
			}
			else
			{
				dCtrlInterp = "X";
			}
			//
			if (!includeDCtrl)
			{
				return antiDInterp;
			}
			else if (dCtrlInterp == "X")
			{
				return "XX";
			}
			else if (antiDInterp == "P" && dCtrlInterp == "N")
			{
				return "P";
			}
			else if (antiDInterp == "N" && dCtrlInterp == "N")
			{
				return "N";
			}
			//BR_41.24
			else if ((antiDInterp == "P" && dCtrlInterp == "P")||(antiDInterp == "N" && dCtrlInterp == "P"))
			{
				return "I";
			}
			else
			{
				return "X";
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/19/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3274"> 
		///		<ExpectedInput>Data rows containing valid Weak D typing test results</ExpectedInput>
		///		<ExpectedOutput>System interpretation of test results</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3275"> 
		///		<ExpectedInput>Data row containing invalid Weak D typing test results ('Q')</ExpectedInput>
		///		<ExpectedOutput>Null Reference Exception thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Find the interpretation of testing results for WeakD
		/// </summary>
		/// <param name="dr">Weak D test results data row</param>
		/// <param name="antiDInterp">Anti-D Interpretation</param>
		/// <param name="DControlInterp">D Control Interpretation</param>
		/// <returns>System Weak D interpretation</returns>
		public static string InterpretWeakDResults(DataRow dr, ref string antiDInterp, ref string DControlInterp)
		{
			return BOL.AntigenTyping.InterpretWeakDResults(dr[ARTIFICIAL.D37].ToString(),dr[ARTIFICIAL.Dc37].ToString(),dr[ARTIFICIAL.DAHG].ToString(),dr[ARTIFICIAL.DcAHG].ToString(),dr[ARTIFICIAL.DCC].ToString(),dr[ARTIFICIAL.DcCC].ToString(),dr[ARTIFICIAL.Interp].ToString(),(dr[ARTIFICIAL.DControlReagentGuid].ToString()!=Guid.Empty.ToString()), ref antiDInterp, ref DControlInterp);
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/19/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3276"> 
		///		<ExpectedInput>Data table containing valid Weak D typing test results</ExpectedInput>
		///		<ExpectedOutput>System interpretation of test results</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3277"> 
		///		<ExpectedInput>Empty data table</ExpectedInput>
		///		<ExpectedOutput>ArgumentException thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Interpret test results for WeakD typing. Returns interpretation and error message.
		/// </summary>
		/// <param name="dt">WeakD data table containing Anti-D testing results and/or D-Control testing results</param>
		/// <returns>Error message</returns>
		public static string InterpretWeakDResults(DataTable dt)
		{
			string SystemInterp = string.Empty;
			string TechInterp = string.Empty;
			string D37 = string.Empty;
			string DAHG = string.Empty;
			string DCC = string.Empty;
			string DCtrl37 = string.Empty;
			string DCtrlAHG = string.Empty;
			string DCtrlCC = string.Empty;
			string antiDInterp = string.Empty;
			string DControlInterp = string.Empty;


			if ((!dt.Columns.Contains(ARTIFICIAL.C37))||
				(!dt.Columns.Contains(ARTIFICIAL.AHG))||
				(!dt.Columns.Contains(ARTIFICIAL.CC)) ||
				(!dt.Columns.Contains(ARTIFICIAL.Interp)) ||
				(!dt.Columns.Contains(Common.DatabaseConstants.ArtificialColumnNames.Indicator)))
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InsufficientColumns(
					MethodBase.GetCurrentMethod().Name).ResString);
			}
			//
			if (dt.Rows.Count > 0)
			{
				DataRow dr = dt.Rows[0];
				D37 = dr[ARTIFICIAL.C37].ToString();
				DAHG = dr[ARTIFICIAL.AHG].ToString();
				DCC = dr[ARTIFICIAL.CC].ToString();
				TechInterp = dr[ARTIFICIAL.Interp].ToString();
			}
			if (dt.Rows.Count > 1)
			{
				DataRow dr = dt.Rows[1];
				DCtrl37 = dr[ARTIFICIAL.C37].ToString();
				DCtrlAHG = dr[ARTIFICIAL.AHG].ToString();
				DCtrlCC = dr[ARTIFICIAL.CC].ToString();
			}
			//
			SystemInterp = BOL.AntigenTyping.InterpretWeakDResults(D37,DCtrl37,DAHG,DCtrlAHG,DCC,DCtrlCC,TechInterp,(dt.Rows.Count == 2), ref antiDInterp, ref DControlInterp);
			//
			//final result based on comparison of system and tech interpretations
			if (SystemInterp == string.Empty)
			{
				foreach(DataRow dr in dt.Rows)
				{
					dr[Common.DatabaseConstants.ArtificialColumnNames.Indicator] = string.Empty;
				}
				return string.Empty;
			}
			else if (SystemInterp == "X")
			{
				foreach(DataRow dr in dt.Rows)
				{
					dr[Common.DatabaseConstants.ArtificialColumnNames.Indicator] = "E";
				}
				return Common.StrRes.Tooltips.Common.InvalidTestResults().ResString;
			}
			if (SystemInterp == "XX")
			{
				foreach(DataRow dr in dt.Rows)
				{
					dr[Common.DatabaseConstants.ArtificialColumnNames.Indicator] = "E";
				}
				return Common.StrRes.Tooltips.Common.InvalidDControl().ResString;
			}
			if (SystemInterp != TechInterp)
			{
				foreach(DataRow dr in dt.Rows)
				{
					dr[Common.DatabaseConstants.ArtificialColumnNames.Indicator] = "E";
				}
				return Common.StrRes.Tooltips.Common.InterpretationNotMatch().ResString;
			}
			else
			{
				foreach(DataRow dr in dt.Rows)
				{
					dr[Common.DatabaseConstants.ArtificialColumnNames.Indicator] = "V";
				}
				return string.Empty;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/19/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3278"> 
		///		<ExpectedInput>Valid blood units with product type of RBC and Plasma</ExpectedInput>
		///		<ExpectedOutput>True indicator for RBC, false for Plasma</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3279"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns a boolean based on whether the unit can be antigen-typed
		/// Implements BR_41.10
		/// </summary>
		/// <param name="unit">Blood unit</param>
		/// <returns>Indicates whether unit can be selected for antigen typing</returns>
		public static bool CanUnitBeAntigenTyped(BOL.BloodUnit unit)
		{
			Common.ProductType type = Common.Utility.GetProductTypeFromProductTypeCode(unit.ProductType.ProductTypeCode);
			//BR_41.10
			if (type == Common.ProductType.WholeBlood ||
				type == Common.ProductType.RedBloodCells ||
				type == Common.ProductType.WashedRedBloodCells ||
				type == Common.ProductType.FrozenRedBloodCells ||
				type == Common.ProductType.FrozenRejuvenatedRedBloodCells ||
				type == Common.ProductType.DeglycerolizedRedBloodCells ||
				type == Common.ProductType.DeglycerolizedRejuvenatedRedBloodCells ||
				type == Common.ProductType.RejuvenatedRedBloodCells ||
				type == Common.ProductType.ApheresisRedBloodCells ||
				type == Common.ProductType.WashedApheresisRedBloodCells ||
				type == Common.ProductType.FrozenApheresisRedBloodCells ||
				type == Common.ProductType.DeglycerolizedApheresisRedBloodCells ||
				type == Common.ProductType.RejuvenatedApheresisRedBloodCells ||
				type == Common.ProductType.FrozenRejuvenatedApheresisRedBloodCells ||
				type == Common.ProductType.DeglycerolizedRejuvenatedApheresisRedBloodCells)
			{
				return true;
			}
			return false;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/19/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3280"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="226"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///	
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns a data table of AntigenTypingTests
		/// </summary>
		/// <returns></returns>
		public static DataTable GetAntigenTypingTests()
		{
			return DAL.ReagentType.GetAntigenTypingTests();
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/19/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3282"> 
		///		<ExpectedInput>
		///		Data rows containing valid antigen typing test results.
		///		1st row contains Rh negative unit and positive Weak D interp.
		///		2nd row contains Rh positivie unit and negative Weak D interp.
		///		</ExpectedInput>
		///		<ExpectedOutput>True for 1st row, false for 2nd row</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3283"> 
		///		<ExpectedInput>Antigen typing new data row with invalid column names</ExpectedInput>
		///		<ExpectedOutput>ArgumentException thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns a boolean as to whether the PhenotypeResults are WeakDPositiveOn a RhNegativeUnit
		/// Implements BR_41.22
		/// </summary>
		/// <param name="drPhenotypeResults"></param>
		/// <param name="interp"></param>
		/// <returns></returns>
		public static bool WeakDPositiveOnRhNegativeUnit(DataRow drPhenotypeResults, string interp)
		{
			if ((!drPhenotypeResults.Table.Columns.Contains(TABLE.OrderableTest.OrderableTestId))||
				(!drPhenotypeResults.Table.Columns.Contains(TABLE.BloodUnit.BloodUnitGuid)))
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InsufficientColumns(
					MethodBase.GetCurrentMethod().Name).ResString);
			}
			//
			//BR_41.22
			//
			bool weakDPos = false;
			//
			if (Convert.ToInt32(drPhenotypeResults[TABLE.OrderableTest.OrderableTestId]) == (int)Common.OrderableTest.AGwD)
			{
				BOL.BloodUnit unit = new BOL.BloodUnit((Guid)drPhenotypeResults[TABLE.BloodUnit.BloodUnitGuid]);
				if (unit.BloodUnitMedia.AboRh.RH == Common.RH.Negative)
				{
					if (interp.Trim().ToUpper() == "P" ||
						interp.Trim().ToUpper() == "I")  //CR 2856
					{
						weakDPos = true;
					}
				}
			}
			//
			return weakDPos;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/19/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3284"> 
		///		<ExpectedInput>Valid data row containing POS antigen typing results on unit with NEG antigen typing results</ExpectedInput>
		///		<ExpectedOutput>Pos result indicates that results conflict with previous results for unit</ExpectedOutput>
		///	</Case>
		///	
		///	
		///<Case type="0" testid ="7028"> 
		///		<ExpectedInput>Valid data row containing NEG antigen typing results on unit with NEG antigen typing results</ExpectedInput>
		///		<ExpectedOutput>Neg result indicates that no results conflict with previous results for unit</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3285"> 
		///		<ExpectedInput>Invalid DataRow</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="7026"> 
		///		<ExpectedInput>Valid data row containing POS antigen typing results on unit with NEG blood unit antigens</ExpectedInput>
		///		<ExpectedOutput>Pos result indicates that results conflict with previous results for unit</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7027"> 
		///		<ExpectedInput>Valid data row containing NEG antigen typing results on unit with NEG blood unit antigens</ExpectedInput>
		///		<ExpectedOutput>Neg result indicates that no results conflict with previous results for unit</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// AntigenTypingMismatch
		/// Implements BR_41.12
		/// </summary>
		/// <param name="drPhenotypeResults"></param>
		/// <param name="interp"></param>
		/// <returns></returns>
		public static bool AntigenTypingMismatch(DataRow drPhenotypeResults, string interp)
		{
			if ((!drPhenotypeResults.Table.Columns.Contains(TABLE.OrderableTest.OrderableTestId))||
				(!drPhenotypeResults.Table.Columns.Contains(TABLE.BloodUnit.BloodUnitGuid))||
				(!drPhenotypeResults.Table.Columns.Contains(TABLE.VamcDivision.DivisionCode)) ||
				(!drPhenotypeResults.Table.Columns.Contains(ARTIFICIAL.InterpGuid)) ||
				(!drPhenotypeResults.Table.Columns.Contains(ARTIFICIAL.Interp)))
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InsufficientColumns(
					MethodBase.GetCurrentMethod().Name).ResString);
			}
			//
			//BR_41.12
			//
			bool mismatch = false;
			//
			DataTable testTypes = BOL.BloodTestType.GetBloodTestTypes(drPhenotypeResults[TABLE.OrderableTest.OrderableTestId].ToString());
			DataRow[] interpTest = testTypes.Select(TABLE.BloodTestType.BloodTestName + " LIKE '*Interp'");
			if (interpTest.Length >0)
			{
				int testTypeID = Convert.ToInt32(interpTest[0][TABLE.BloodTestType.BloodTestTypeId]);
				//
				DataTable unitTests = BOL.BloodUnitTest.GetBloodUnitTestResult((Guid)drPhenotypeResults[TABLE.BloodUnit.BloodUnitGuid],testTypeID,Common.LogonUser.LogonUserDivisionCode);
				DataRow[] otherTests = unitTests.Select(TABLE.BloodUnitTest.BloodUnitTestGuid + " <> '" + drPhenotypeResults[ARTIFICIAL.InterpGuid].ToString() + "'");
				foreach (DataRow unitTest in otherTests)
				{
					string testResult = unitTest[TABLE.BloodUnitTest.TestResultId].ToString().Trim();
					if ((testResult != "I") && (testResult != "") && (testResult != interp.Trim()))
					{
						return true;
					}
				}
				//
				DataTable antigenTests = BOL.AntibodyType.GetAntigenTest();
				DataRow[] matchingAntigenTest = antigenTests.Select("BloodTestTypeId = " + Convert.ToInt32(interpTest[0][TABLE.BloodTestType.BloodTestTypeId]));
				if (matchingAntigenTest.Length > 0)
				{
					int antigenTypeID = Convert.ToInt32(matchingAntigenTest[0][TABLE.AntigenType.AntigenTypeId]);
					DataTable unitAntigens = BOL.BloodUnitAntigen.GetBloodUnitAntigens((Guid)drPhenotypeResults[TABLE.BloodUnit.BloodUnitGuid],Common.LogonUser.LogonUserDivisionCode);
					DataRow[] specificAntigens = unitAntigens.Select("AntigenTypeId = " + antigenTypeID);
					foreach (DataRow unitAntigen in specificAntigens)
					{
						bool posResult = (bool)unitAntigen[TABLE.BloodUnitAntigen.Positive];
						if ((posResult && (interp.Trim() == "N")) ||
							(!posResult && (interp.Trim() == "P")))
						{
							return true;
						}
					}
				}
			}
			//
			return mismatch;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/19/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6766"> 
		///		<ExpectedInput>Guid of unit issued to patient with Anti-C transfusion requirement, AGC orderable test ID, Pos interp</ExpectedInput>
		///		<ExpectedOutput>Typing mismatch indicator of true returned</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="6872"> 
		///		<ExpectedInput>Guid of unit issued to patient with K antigen-negative requirement, AGK orderable test ID, Pos interp</ExpectedInput>
		///		<ExpectedOutput>Typing mismatch indicator of true returned</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="6870"> 
		///		<ExpectedInput>Neg interp</ExpectedInput>
		///		<ExpectedOutput>Typing mismatch indicator of false returned</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="7032"> 
		///		<ExpectedInput>Guid of unit issued to Rh Neg patient, AGwD orderable test ID, Pos interp</ExpectedInput>
		///		<ExpectedOutput>Typing mismatch indicator of true returned</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6871"> 
		///		<ExpectedInput>Non-issued blood unit Guid</ExpectedInput>
		///		<ExpectedOutput>Typing mismatch indicator of false returned</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7033"> 
		///		<ExpectedInput>Guid of unit issued to patient with no ABIDs or TRs, AGN orderable test ID, Pos interp</ExpectedInput>
		///		<ExpectedOutput>Typing mismatch indicator of false returned</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7038"> 
		///		<ExpectedInput>Guid of unit issued to Rh Pos patient, AGwD orderable test ID, Pos interp</ExpectedInput>
		///		<ExpectedOutput>Typing mismatch indicator of false returned</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="0" testid ="9159"> 
		///		<ExpectedInput>Guid of unit issued to patient with Anti-C transfusion requirement coming from conversion, AGC orderable test ID, Pos interp</ExpectedInput>
		///		<ExpectedOutput>Typing mismatch indicator of true returned</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Indicates if current testing results in a mismatch with issued patient
		/// Implements BR_56.29 for UC_41
		/// </summary>
		/// <param name="bloodUnitGuid"></param>
		/// <param name="orderableTestId"></param>
		/// <param name="interp"></param>
		/// <returns></returns>
		public static bool PatientAntigenTypingMismatch(Guid bloodUnitGuid, int orderableTestId, string interp)
		{
			if (interp.ToString().Trim() != "P")
			{
				//Test result not positive
				return false;
			}
			//
			DataTable issuedUnit = DAL.IssuedUnit.GetIssuedUnitByBloodUnitGuid(bloodUnitGuid);
			if (issuedUnit.Rows.Count == 0)
			{
				//Unit not issued to a patient
				return false;
			}
			//
			BOL.Patient issuedPatient = new BOL.Patient((Guid)issuedUnit.Rows[0][TABLE.Patient.PatientGuid]);
			//
			//For Weak D, just test if the patient is Rh negative
			if (orderableTestId == (int)Common.OrderableTest.AGwD)
			{
				if (issuedPatient.AboRh.RH == Common.RH.Negative)
				{
					return true;
				}
				else
				{
					return false;
				}
			}
			//
			DataTable dtBloodTestTypes = BOL.BloodTestType.GetBloodTestTypes(orderableTestId.ToString());
			DataRow[] drBloodTestTypes = dtBloodTestTypes.Select(TABLE.BloodTestType.BloodTestName + " LIKE '%Interp'");
			DataTable dtAntigenTests = BOL.AntibodyType.GetAntigenTest();
			DataRow[] drAntigenTests = dtAntigenTests.Select(TABLE.AntigenTest.BloodTestTypeId + " = '"+drBloodTestTypes[0][TABLE.BloodTestType.BloodTestTypeId].ToString()+"'");
			int antigenTypeId = Convert.ToInt32(drAntigenTests[0][TABLE.AntigenTest.AntigenTypeId]);
			//
			// patient's antigen negative requirements - from UC_38
			DataRow [] drPatientAntigens = issuedPatient.TransfusionRequirements.Select(TABLE.PatientTransfusionRequirement.TransfusionRequirementCategoryCode + " = '" + Common.Utility.GetTransfusionCategoryCodeFromEnum(Common.TransfusionRequirementCategoryCode.AntigenNegative) + "' AND RecordStatusCode = '" + Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active) + "'");
			//
			foreach(DataRow drAntigenReq in drPatientAntigens)
			{
				if ((int)drAntigenReq[TABLE.AntigenType.AntigenTypeId]==antigenTypeId)
				{
					//Patient has an antigen negative requirement and unit is positive for antigen
					//this is a mismatch
					return true;
				}
			}
			//
			DataTable dtAntibodyTypes = BOL.AntibodyType.GetAntibodiesForAntigenType(antigenTypeId);
			//
			// patient's clinically significant antibodies - from UC_38
			foreach (DataRow drPatientAntibody in issuedPatient.ClinicallySignificantAntibodies.Rows)
			{
				// Do not generate Antigen negative requirements for these.
				if (!System.Enum.IsDefined(typeof(Common.ExcludedAntibodies), (int) drPatientAntibody[TABLE.AntibodyType.AntibodyTypeId]))
				{
					//
					foreach(DataRow drAntibodyType in dtAntibodyTypes.Rows)
					{
						if ((int)drAntibodyType[TABLE.AntibodyType.AntibodyTypeId]==(int) drPatientAntibody[TABLE.AntibodyType.AntibodyTypeId])
						{
							//Patient is positive for antibody and unit is positive for antigen
							//this is a mismatch
							return true;
						}
					}
				}
			}
			//
			//Patients has no antibodies or antigen negative requirement mismatches
			return false;

		}

		#endregion
	}
}
